/***************************************************************************************\
*   Name         : Test_Code                                                           
*   Author       : Pascal Krommendijk, Joao Pereira                                    
*   Created      : 19 May, 2019                                                        
*   Last Modified: 13 June, 2019                                                        
*   Version      : 0.4                                                                 
*   Overview     : Preparations
*                  Setup() 
*                  loop()
*                  Menu Options Functions
*                  Menu Fundamentals Functions               
*                  Rest Functions (evaluating buttons, loadingscreens, checking Conditions)
*                  Diagnostic Functions
*   
*   Notes        : Requirements needed:                                                
*                  Libraries: LiquidCrystal, Wire, Servo, DHT                         
*                                                                                      
*   To Do        : Make the Code beautiful                                           
*                  Diagnostics implement                                               
*                       - Test the Battery Measurement 
*                       - all test functions
*                  use blinkLEDs() when busy, ready, error etc. (--------)
****************************************************************************************/

/**********Tweak Section**********/
#define maxTemperatureTweak 28
#define maxHumidityTweak 80
#define TemperatureTweak 0         //if there is a unprecision you can fix it by subtracting or adding here (eg. +2/-2)
#define temperatureMarginTweak 7   //the margin in Celcius where the fan and servo turn off after turning on and decreasing temperature
#define openServoAngleTweak 0      //test this out with different angles etc.
#define closeServoAngleTweak 0
#define firstShowHours 28

#define SecondsInProcessHourTweak 3600 //3600 = 1 hour normally
#define LoadingSpeedTweak 25       //e.g: 2 seconds in total => 2.000/80 = 25 (delay between each Line) 
#define watchTimeDelayTweak 2000
#define watchTimeBetweenScrollTweak 1000
#define AmountOfSecondsBeforePowerSavingTweak 60
#define AntiFlikkeringDelayTweak 30

String menuItems[] = {"Start drying","Process Status","Conditions","Diagnostics","Stand-By","Stop Process","About.."};
/*********************************/

#include <Wire.h>
#include <LiquidCrystal.h>
#include <Servo.h>
#include "DHT.h"

/*************PIN Preparation******************/
#define BatteryPin A1
#define Green A2
#define Orange A3
#define Red A4
#define BackLight 10
#define FanPower 11
#define ServoPower 12
#define ServoMotorPin 13

/************Servo Preparation*****************/
Servo lilMotor;
int ServoFanOn, ServoOn;
int openVentAngle = 90; //needs to be defined
int closeVentAngle = 0; // " " " "

/************MENU Preparation******************/
//put MenuItems here after tweaking----------------------------------
int MenuFactor = 2, cursorPosition = 0, currentMenuPage = 0;
int maxMenuPages =(sizeof(menuItems) / sizeof(String)) - MenuFactor;

/****************LCD Preparation**************/
LiquidCrystal LCD(8, 9, 4, 5, 6, 7);
  //Loading Bar
  byte p1[8]        = {0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10}; // equals |
  byte p2[8]        = {0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18}; // equals ||
  byte p3[8]        = {0x1C,0x1C,0x1C,0x1C,0x1C,0x1C,0x1C,0x1C}; // equals |||
  byte p4[8]        = {0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E}; // equals ||||
  byte p5[8]        = {0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F}; // equals |||||
  byte downArrow[8] = {0b00100, 0b00100, 0b00100, 0b00100, 0b00100, 0b10101, 0b01110, 0b00100};
  byte upArrow[8]   = {0b00100, 0b01110, 0b10101, 0b00100, 0b00100, 0b00100, 0b00100, 0b00100};
  byte menuCursor[8]= { B01000,  B00100,  B00010,  B00001,  B00010,  B00100,  B01000,  B00000};

/***************DHT Preparation***************/
#define DHTPINTOP 3
#define DHTPINBOTTOM 2
#define DHTTYPE DHT22
DHT dhtTop(DHTPINTOP, DHTTYPE);
DHT dhtBottom(DHTPINBOTTOM, DHTTYPE);

/*************Process Preparation*************/
int processMode = 0;                    //0 = nothing is running (standby and diagnostics are availible) 
                                        //1 = a program is running (standby & diagnostics are unavailible) 
long endTime = 0;                       //End time of the process. Is current time plus programlength
long ellapsedTimeH = 0;                 //ellapsed Time in hours 
long ellapsedTimeM = 0;                 // ""           in minutes  
int  selectedHours = 0;                        
long processTime   = 0;                 //Time in milliseconds: hours *3600 seconds* 1000 milliseconds

/***************Energy Preparation************/
long energySavingCounter = 0;           
long powerSavingTime = 1000L * AmountOfSecondsBeforePowerSavingTweak;
float maximumTemperature = maxTemperatureTweak;
float maximumHumidity = maxHumidityTweak;

void setup() {
  //Serial.begin(9600);
  
  /*****LCD Setup*****/
  LCD.begin(16, 2);
  LCD.createChar(0, menuCursor);
  LCD.createChar(1, upArrow);
  LCD.createChar(2, downArrow);
  LCD.createChar(3, p1);
  LCD.createChar(4, p2);
  LCD.createChar(5, p3);
  LCD.createChar(6, p4);
  LCD.createChar(7, p5);
  LCD.clear();

  /*****Sensor Setup*****/
  pinMode(DHTPINTOP, INPUT); //if 10k external Resistor is used -->only INPUT
  pinMode(DHTPINBOTTOM, INPUT);
  dhtTop.begin();
  dhtBottom.begin();

  /*****Digital Pins Setup*****/
  pinMode(Green,    OUTPUT);
  pinMode(Orange,   OUTPUT);
  pinMode(Red,      OUTPUT);
  pinMode(BackLight,OUTPUT);
  digitalWrite(BackLight, HIGH);
  pinMode(FanPower,  OUTPUT);
  pinMode(ServoPower,OUTPUT);
  pinMode(BatteryPin, INPUT);

  /*****Servo Motor Setup*****/
  digitalWrite(ServoPower, HIGH);
  delay(2000);
  lilMotor.attach(ServoMotorPin);
  //digitalWrite(ServoPower, LOW);

  
  //Loading Bar for estatics
  LoadingScreen(0); //0 = start up
}

void loop() {
  if(processMode == 0){
    //Process Mode off
    mainMenuDraw();
    drawCursor();
    operateMainMenu();
  }
  else{
    //Process Mode on
    checkingConditions(maximumTemperature, maximumHumidity,endTime);
    analogWrite(A1, 1023);
    mainMenuDraw();
    drawCursor();
    operateMainMenu();
  }
}


/********************The Different Menu Options*********************/

void ChooseProcess() { 
  int backButton = 0;   //used to get out of this submenu
  
  selectedHours = firstShowHours;
  if(processMode != 1){
    while (backButton == 0) {
      delay(AntiFlikkeringDelayTweak);
      LCD.clear();
      LCD.write(byte(1));
      LCD.print(" +1h Time: ");
      LCD.print(selectedHours);
      LCD.print(" h");
      LCD.setCursor(0,1);
      LCD.write(byte(2));
      LCD.print(" -1h");
      
      switch (evaluateButton()) {
        case 0: // no action taken / no button pressed
          break;
          
        /*********"right" button pressed*********/
        case 1: 
          processTime = selectedHours * SecondsInProcessHourTweak * 1000L; //total process time in milliseconds
          endTime = millis()+processTime;                   //in milliseconds, will be converted in the next function
          
          //LCD gives sign to start Program
          LCD.clear();
          LCD.print("Great choice!");
          LCD.setCursor(0,1);
          LCD.print("See you in ");
          LCD.print(selectedHours);    //set timer in currently shown hours
          LCD.print(" h");
          delay(watchTimeDelayTweak);
                    
          backButton = 1; 
          processMode = 1; 
          break;
        
        /*********"up" button pressed*********/
        case 2: 
          selectedHours++;
          backButton = 0; //not out of the submenu
          break;
  
        /******"down" button pressed******/
        case 3: 
          if(selectedHours > 1){
          selectedHours--;
          }
          backButton = 0; //not out of the submenu
          break;
        case 4: 
          backButton = 1; //out of the sub menu but no Program mode
          break;
      }
    }
  }
  else{
    LCD.clear();
    LCD.print("Stop current");
    LCD.setCursor(0,1);
    LCD.print("Program in order");
    delay(watchTimeBetweenScrollTweak);
    LCD.clear();
    LCD.print("Program in order");
    LCD.setCursor(0,1);
    LCD.print("to start another");
    delay(watchTimeDelayTweak);
  }
}

void ProcessStatus() { 
  int backButton = 0;
  
  LCD.clear();
  while(evaluateButton() != 4 && backButton != 1) {
    LCD.setCursor(0,0);
    if(processMode == 0){
      LCD.print("No Program");
      LCD.setCursor(0,1);
      LCD.print("selected");
      backButton = 1;
      delay(watchTimeDelayTweak);
    }
    else{
      ellapsedTimeH = ((endTime - millis())/SecondsInProcessHourTweak)/1000L;
      ellapsedTimeM = (((endTime - millis())/60)/1000L) -(ellapsedTimeH *60);
      LCD.print("Chosen Time:");
      LCD.print(selectedHours);
      LCD.print(" h");
      LCD.setCursor(0,1);
      LCD.print(ellapsedTimeH);
      LCD.print(" h ");
      LCD.print(ellapsedTimeM);
      LCD.print(" min left");
      delay(AntiFlikkeringDelayTweak);
    }
  }
}

void Conditions() {
  int backButton = 0;
  int sensorPosition = 0;
  LCD.clear();
  while (backButton == 0) {
    //able to present both informations from both DHT22 sensors 
    //by changing the sensor that wants to be presented
    
    if(sensorPosition == 1){
      float tempTop = dhtTop.readTemperature();
      float humidTop = dhtTop.readHumidity();
      //checks if sensor is connected and gives "a" measurement
      if(tempTop > -40 && tempTop < 80 || humidTop > 0 && humidTop < 100){ 
        LCD.setCursor(0,0);
        LCD.print("Heat : ");
        LCD.print(tempTop);
        LCD.print(" C");
        LCD.setCursor(15,0);
        LCD.write(byte(1));
        LCD.setCursor(0,1);
        LCD.print("Humid: ");
        LCD.print(humidTop);
        LCD.print(" %"); 
        //analogWrite(A2);
      }
      else{
        
        LCD.print("Upper Sensor");
        LCD.setCursor(0,1);
        LCD.print("not connected.  ");
        //switch on warning Led-------------------------------------------------------------------------------------------------
      }
    }
    else if(sensorPosition == 0){
      float tempBottom = dhtBottom.readTemperature();
      float humidBottom = dhtBottom.readHumidity();
      //checks if sensor is connected and gives "a" measurement
      if(tempBottom > -40 && tempBottom < 80 || humidBottom > 0 && humidBottom < 100){
        LCD.setCursor(0,0);
        LCD.print("Heat : ");
        LCD.print(tempBottom);
        LCD.print(" C");
        LCD.setCursor(15,1);
        LCD.write(byte(2));
        LCD.setCursor(0,1);
        LCD.print("Humid: ");
        LCD.print(humidBottom);
        LCD.print(" %"); 
        //switch off warning Led-------------------------------------------------------------------------------------------------
      }
      else{
        LCD.print("Lower Sensor");
        LCD.setCursor(0,1);
        LCD.print("not connected.  ");
        //switch on warning Led-------------------------------------------------------------------------------------------------
      }
    }
    
    /*******Evaluate the button stuff*******/
    switch (evaluateButton()) {
      case 2: //up button
        sensorPosition = 1; //up button for upper sensor
        LCD.clear();
        break;
      case 3: //down button
        sensorPosition = 0; //down button for lower sensor
        LCD.clear();
        break;
      case 4:  //Back-button is pressed
        backButton = 1;
        break;
    }
  }
}

void Diagnostics() {  
  if((endTime-millis()) >= 120000){
    LCD.clear();
    LCD.setCursor(3, 0);
    LCD.print("Diagnostics");
    LCD.setCursor(0,1);
   
    /*******Testing the Servo Motor*******/
    testMotorAndFan();
    /********Testing both Sensors*********/
    testSensors();
    /*******Testing Battery Voltage*******/
    testBattery();
    
    /*******Testing external LEDs*********/
    testLEDs();
    /***********Testing the LCD***********/
    testLCD();    
 LCD.clear();
  LCD.setCursor(0,0);
  LCD.print("Diagnostic scan");
  LCD.setCursor(0,1);
  LCD.print("is completed");
  delay(watchTimeDelayTweak);
  }
  else{
    LCD.setCursor(0,0);
    LCD.print("Progress is close");
    LCD.setCursor(0,1);
    LCD.print("to finish..");
    delay(watchTimeDelayTweak);
  }
}

void StandBy() { 
  //Standby is not possible while a process is running
  if(processMode == 0){
  //Use LoadingScreen for Standby animation
  digitalWrite(FanPower, LOW);
  turnServoMotor(closeVentAngle);
  digitalWrite(ServoPower, LOW);
  LoadingScreen(1);
  digitalWrite(BackLight, LOW);
 
  }
  else{
    LCD.clear();
    LCD.print("DryFoo is");
    LCD.setCursor(0,1);
    LCD.print("busy right now");
    delay(watchTimeDelayTweak);
    
  }
  //Any Button press will bring the device out of stand by mode
  //because 0 = nothing is pressed
  while (evaluateButton() == 0 && processMode == 0) {}
     digitalWrite(BackLight, HIGH);
     LoadingScreen(0);   
}

void StopProcess() { 
  if(processMode == 0){
      LCD.clear();
    LCD.print("There's nothing");
    LCD.setCursor(0,1);
    LCD.print("to cancel");
  }
  else{
    LCD.clear();
    LCD.print("Process has been");
    LCD.setCursor(0,1);
    LCD.print("cancelled!");
    processMode = 0;
    processTime = 0;
  }
  delay(watchTimeDelayTweak);
}

void AboutDisplay() { 
    LCD.clear();
    LCD.print("DryFoo Ver: 1.0");
    LCD.setCursor(0,1);
    LCD.print("Product by Besol");
    delay(watchTimeDelayTweak);
}


/***************************Menu Fundamentals***************************/

// This function will generate the 2 menu items that can fit on the screen. 
// They will "move up" as you scroll through the menu. 
// Up and down arrows will indicate your current menu position (top, middle, down.
void mainMenuDraw() {
  LCD.clear();
  LCD.setCursor(1, 0);
  LCD.print(menuItems[currentMenuPage]);
  LCD.setCursor(1, 1);
  LCD.print(menuItems[currentMenuPage + 1]);
  
  if (currentMenuPage == 0) {      //First Page Arrow
    LCD.setCursor(15, 1);
    LCD.write(byte(2));     //print Down Arrow
  } 
  else if (currentMenuPage > 0 and currentMenuPage < maxMenuPages) { //Middle Page Arrows
    LCD.setCursor(15, 1);
    LCD.write(byte(2));     //print Down Arrow
    LCD.setCursor(15, 0);
    LCD.write(byte(1));     //print UP Arrow
  } 
  else if (currentMenuPage == maxMenuPages) { //Last Page Arrow
    LCD.setCursor(15, 0);
    LCD.write(byte(1));     //print Down Arrow
  }
}

// When called, this function will erase the current cursor 
// And redraw it based on the cursorPosition and currentMenuPage variables.
void drawCursor() {
  
  // Erases current cursor
  for (int x = 0; x < 2; x++) {     
    LCD.setCursor(0, x);
    LCD.print(" ");
  }
  // The menu is set up to be progressive 
  //   menuPage 0 = Item 1 & Item 2
  //   menuPage 1 = Item 2 & Item 3
  //   menuPage 2 = Item 3 & Item 4
  // so in order to determine where the cursor should be 
  // you need to see if you are at an odd or even menu page and an odd or even cursor position.
  if (currentMenuPage % 2 == 0) {
    if (cursorPosition % 2 == 0) {  // If the menu page is even and the cursor position is even that means the cursor should be on line 1
      LCD.setCursor(0, 0);
      LCD.write(byte(0));
    }
    if (cursorPosition % 2 != 0) {  // If the menu page is even and the cursor position is odd that means the cursor should be on line 2
      LCD.setCursor(0, 1);
      LCD.write(byte(0));
    }
  }
  if (currentMenuPage % 2 != 0) {
    if (cursorPosition % 2 == 0) {  // If the menu page is odd and the cursor position is even that means the cursor should be on line 2
      LCD.setCursor(0, 1);
      LCD.write(byte(0));
    }
    if (cursorPosition % 2 != 0) {  // If the menu page is odd and the cursor position is odd that means the cursor should be on line 1
      LCD.setCursor(0, 0);
      LCD.write(byte(0));
    }
  }
}

//Function to use the Menu with the buttons
//actions are taken in this function
void operateMainMenu() {
  int activeButton = 0;
  
  int button;
  long compareTime = millis()+powerSavingTime;
  while (activeButton == 0) {
    if(millis() >= endTime){
    processMode = 0;
    }
    
    button = evaluateButton();
    switch (button) {
      case 0: 
        // energy saving after a couple minutes
        energySavingCounter = millis();
        break;
      /*********"right" button pressed*********/
      case 1: 
        button = 0;
        energySavingCounter = 0;
        compareTime = millis()+powerSavingTime;
        digitalWrite(BackLight, HIGH);
        switch (cursorPosition) { 
          case 0:
            ChooseProcess();
            break;
          case 1:
            ProcessStatus();
            break;
          case 2:
            Conditions();
            break;
          case 3:
            Diagnostics();
            break;
          case 4:
            StandBy();
            break;
          case 5:
            StopProcess();
            break;
          case 6:
            AboutDisplay();
            break;  
        }
        activeButton = 1;

        //update Menu again by calling these
        mainMenuDraw();
        drawCursor();
        break;
      
      /*********"up" button pressed*********/
      case 2: 
        button = 0;
        energySavingCounter = 0;
        compareTime = millis()+powerSavingTime;
        if (currentMenuPage == 0) {
          cursorPosition = cursorPosition - 1;
          cursorPosition = constrain(cursorPosition, 0, ((sizeof(menuItems) / sizeof(String)) - 1));
        }
        if (currentMenuPage % 2 == 0 and cursorPosition % 2 == 0) {
          currentMenuPage = currentMenuPage - 1;
          currentMenuPage = constrain(currentMenuPage, 0, maxMenuPages);
        }

        if (currentMenuPage % 2 != 0 and cursorPosition % 2 != 0) {
          currentMenuPage = currentMenuPage - 1;
          currentMenuPage = constrain(currentMenuPage, 0, maxMenuPages);
        }

        cursorPosition = cursorPosition - 1;
        cursorPosition = constrain(cursorPosition, 0, ((sizeof(menuItems) / sizeof(String)) - 1));
        activeButton = 1;
        
        //update Menu again by calling these
        mainMenuDraw();
        drawCursor();
        break;

      /******"down" button pressed******/
      case 3: 
        button = 0;
        energySavingCounter = 0;
        compareTime = millis()+powerSavingTime;
        if (currentMenuPage % 2 == 0 and cursorPosition % 2 != 0) {
          currentMenuPage = currentMenuPage + 1;
          currentMenuPage = constrain(currentMenuPage, 0, maxMenuPages);
        }

        if (currentMenuPage % 2 != 0 and cursorPosition % 2 == 0) {
          currentMenuPage = currentMenuPage + 1;
          currentMenuPage = constrain(currentMenuPage, 0, maxMenuPages);
        }
        cursorPosition = cursorPosition + 1;
        cursorPosition = constrain(cursorPosition, 0, ((sizeof(menuItems) / sizeof(String)) - 1));
        activeButton = 1;
        
        //update Menu again by calling these
        mainMenuDraw();
        drawCursor();
        break;
        //Back Button not needed in "main" menu
    }
    //Save energy when an amount of seconds has past (defined in the preparation)
    if(energySavingCounter > compareTime){ 
          digitalWrite(BackLight, LOW);
        }
    else{
          digitalWrite(BackLight, HIGH);
        }
  }
}


/******************************Rest Funtions***************************/

//is used to put one definitive integer to the buttons pressed.
//avoiding some voltage level uncertainty
int evaluateButton() {
  int readKey = analogRead(0);
    if (readKey < 790) {
      delay(50); //debouncing time
      readKey = analogRead(0);
    }
  
  int result = 0;
  if (readKey < 100) {
    result = 1; // right
  } else if (readKey < 300) {
    result = 2; // up
  } else if (readKey < 500) {
    result = 3; // down
  } else if (readKey < 700) {
    result = 4; // left
  }
  return result;
}

//While in processMode the device contiuously 
//compares current values to set maximum values
int checkingConditions(float maxTemp, float maxHumid, long endingTiming){
  if(millis() >= endingTiming){
    processMode = 0;
  }
  else{
    if((dhtTop.readTemperature() >= maxTemp || dhtBottom.readTemperature() >= maxTemp) && ServoFanOn == 0){
      //servo and fan turn on
      digitalWrite(FanPower, HIGH);
      
      turnServoMotor(openVentAngle);
      
      ServoFanOn = 1;
    }
    if((dhtTop.readTemperature() <= (maxTemp-temperatureMarginTweak) || dhtBottom.readTemperature() <= (maxTemp-temperatureMarginTweak)) ){
      //servo and fan turn off
      digitalWrite(FanPower, LOW);
      
      turnServoMotor(closeVentAngle);
      
      ServoFanOn = 0;
    }
    if(dhtTop.readHumidity() >= (maxHumid) || dhtBottom.readHumidity() >= (maxHumid)){
    digitalWrite(ServoPower, HIGH);
    turnServoMotor(openVentAngle);
    digitalWrite(ServoPower, LOW);
    ServoOn = 1;
    }
    if(dhtTop.readHumidity() <= (maxHumid-temperatureMarginTweak) || dhtBottom.readHumidity() <= (maxHumid-temperatureMarginTweak)){
      //servo only turn off
      
      turnServoMotor(closeVentAngle);
      
      ServoOn = 0;
    }
  }
}

//Loading screen for Start up & Stand by
void LoadingScreen(int choice){
  //Choice 0 = Starting Up Screen
  //Choice 1 = Standy By Screen
  //Choice 2 = Process finished Screen
  LCD.clear();
  LCD.setCursor(0,0);
  if(choice == 0){
  LCD.print("Starting Up...");
  for(int inRowPosition = 0; inRowPosition < 16; inRowPosition++){
    for(int inDigitPosition  = 3; inDigitPosition < 8; inDigitPosition++){
      //LCD.clear();
      LCD.setCursor(inRowPosition,1);
      LCD.write(inDigitPosition);
      delay(LoadingSpeedTweak);
    }
  }
   
  delay(watchTimeBetweenScrollTweak/2);
  LCD.clear();
  LCD.setCursor(4,0);
  LCD.print("Welcome!");
  LCD.setCursor(0,1);
  LCD.print("Enjoy your food.");
  delay(watchTimeDelayTweak);
  LCD.clear();
  }
  else if(choice == 1){
    LCD.print("Standing By");
    for(int inRowPosition = 0; inRowPosition < 16; inRowPosition++){
      for(int inDigitPosition  = 3; inDigitPosition < 8; inDigitPosition++){
        //LCD.clear();
        LCD.setCursor(inRowPosition,1);
        LCD.write(inDigitPosition);
        delay(LoadingSpeedTweak);
      }
    }
    delay(watchTimeBetweenScrollTweak/2);
    LCD.clear();
    LCD.setCursor(5,0);
    LCD.print("Bye Bye!");
    delay(watchTimeBetweenScrollTweak);
    LCD.clear();
  }
  else{
    LCD.print("Drying Completed");
    LCD.print("Enjoy your Food!");
    delay(watchTimeDelayTweak);
    //LED flashing stuff------------------------------------------------------------------------
  }

}

//to reduce code with LED visuals
void blinkLED(String Color, int Amount, int OnTime, int OffTime){
  int LEDpin = 6;
  if(Color == "Green"){
    LEDpin = 2;
  }
  else if(Color == "Orange"){
    LEDpin = 3;
  }
  else if(Color == "Red"){
    LEDpin = 4;
  }
  for(int i = 0; i < Amount; i++){
    analogWrite(LEDpin, 1023);
    delay(OnTime);
    analogWrite(LEDpin, 0);
    delay(OffTime);
  }
}

void turnServoMotor(int setAngle){
  int previousAngle = 0;
  digitalWrite(ServoPower, HIGH);
  previousAngle = lilMotor.read();
  if(previousAngle > setAngle){
    for(int i = previousAngle; i >= setAngle; i--){
      lilMotor.write(i);
      delay(20);  
    }
  }
  else{
    for(int i = previousAngle; i <= setAngle; i++){
      lilMotor.write(i);
      delay(20);
    }
  }
  digitalWrite(ServoPower, LOW);
}
/***********************diagnostic Functions***************************/
//function to test the servo motor in diagnostics
void testMotorAndFan(){
  if(processMode == 0){
  turnServoMotor(closeVentAngle);
  int checkAngle = 0;
  int testResult = 1;
  for(int angle = 0; angle < 181; angle++){
    lilMotor.write(angle);
    delay(LoadingSpeedTweak);
    checkAngle = lilMotor.read();
    if(checkAngle != angle){
     angle =190;
     //no good result
      LCD.print("Motor:      Off!");
      delay(watchTimeBetweenScrollTweak);
      LCD.clear();
      LCD.print("Motor:      Off!");
      LCD.setCursor(0,1);
      LCD.print("Fan        Off!");
      delay(watchTimeBetweenScrollTweak);
      LCD.clear();
      LCD.print("Fan        Off!");
      LCD.setCursor(0,1);
    }
    else{
      //fine result
      LCD.print("Motor:       On!");
      delay(watchTimeBetweenScrollTweak);
      LCD.clear();
      LCD.print("Motor:       On!");
      LCD.setCursor(0,1);
      LCD.print("Fan          On!");
      delay(watchTimeBetweenScrollTweak);
      LCD.clear();
      LCD.print("Fan          On!");
      LCD.setCursor(0,1);
    }
  }
  lilMotor.write(0);
  }
  else{
    if(ServoFanOn == 1){
      LCD.print("Motor:       On!");
      delay(watchTimeBetweenScrollTweak);
      LCD.clear();
      LCD.print("Motor:       On!");
      LCD.setCursor(0,1);
      LCD.print("Fan:         On!");
      delay(watchTimeBetweenScrollTweak);
      LCD.clear();
      LCD.print("Fan:         On!");
      LCD.setCursor(0,1);
    }
    else if(ServoOn == 1){
      LCD.print("Motor        On!");
      delay(watchTimeBetweenScrollTweak);
      LCD.clear();
      LCD.print("Motor        On!");
      LCD.setCursor(0,1);
      LCD.print("Fan         Off!");
      delay(watchTimeBetweenScrollTweak);
      LCD.clear();
      LCD.print("Fan         Off!");
      LCD.setCursor(0,1);
    }
  }
}

void testBattery(){
  //use pin A1 and a conversion to %
  int batteryNumber = 0;
  batteryNumber = analogRead(BatteryPin);
  float batteryVoltage = 3.8*(batteryNumber/1023);
  int batteryPercentage = round(100*(batteryVoltage/3.8));
  LCD.setCursor(0,1);
  LCD.print("Battery");
  LCD.setCursor(12,1);
  LCD.print(batteryPercentage);
  LCD.print("%");
  delay(watchTimeBetweenScrollTweak);
  LCD.clear();
  LCD.setCursor(0,0);
  LCD.print("Battery");
  LCD.setCursor(12,0);
  LCD.print(batteryPercentage);
  LCD.print("%");
  LCD.setCursor(0,1);
}

void testSensors(){
  //0 = both working 
  //1 = Bottom not working 
  //2 = Top not Working 
  //3 = Both Not Working
  int Working = 0; 
  float tempBottom = dhtBottom.readTemperature();
  float humidBottom = dhtBottom.readHumidity();
  //checks if sensor is connected and gives "a" measurement
  if(!(tempBottom > -40 && tempBottom < 80 || humidBottom > 0 && humidBottom < 100)){
    Working++;
  }
  
  float tempTop = dhtTop.readTemperature();
  float humidTop = dhtTop.readHumidity();
  //checks if sensor is connected and gives "a" measurement
  if(!(tempTop > -40 && tempTop < 80 || humidTop > 0 && humidTop < 100)){ 
    Working +=2;
  }
  switch(Working){
      case 0:
        LCD.print("Top Sensor  On!");
        delay(watchTimeBetweenScrollTweak);
        LCD.clear();
        LCD.print("Top Sensor  On!");
        LCD.setCursor(0,1);
        LCD.print("Low Sensor  On!");
        delay(watchTimeBetweenScrollTweak);
        LCD.clear();
        LCD.print("Low Sensor  On!");
        break;
      case 1:
        LCD.print("Top Sensor  On!");
        delay(watchTimeBetweenScrollTweak);
        LCD.clear();
        LCD.print("Top Sensor  On!");
        LCD.setCursor(0,1);
        LCD.print("Low Sensor Off!");
        delay(watchTimeBetweenScrollTweak);
        LCD.clear();
        LCD.print("Low Sensor Off!");
        break;
      case 2:
        LCD.print("Top Sensor Off!");
        delay(watchTimeBetweenScrollTweak);
        LCD.clear();
        LCD.print("Top Sensor Off!");
        LCD.setCursor(0,1);
        LCD.print("Low Sensor  On!");
        delay(watchTimeBetweenScrollTweak);
        LCD.clear();
        LCD.print("Low Sensor  On!");
        break;
      case 3:
        LCD.print("Top Sensor Off!");
        delay(watchTimeBetweenScrollTweak);
        LCD.clear();
        LCD.print("Top Sensor Off!");
        LCD.setCursor(0,1);
        LCD.print("Low Sensor Off!");
        delay(watchTimeBetweenScrollTweak);
        LCD.clear();
        LCD.print("Low Sensor Off!");
        break;
          
  }
}

void testLEDs(){
    // put Leds on after each other and ask on LCD
    // if everything lighted up and press button is yes
}

void testLCD(){
  LCD.clear();
  LCD.setCursor(0,0);
  for(int row = 0; row < 2; row++){
    for(int inRowPosition = 0; inRowPosition < 16; inRowPosition++){
      for(int inDigitPosition  = 3; inDigitPosition < 8; inDigitPosition++){
        //LCD.clear();
        LCD.setCursor(inRowPosition,row);
        LCD.write(inDigitPosition);
        delay(LoadingSpeedTweak);
      }
    }
  }
  //adding a button press for if everything lighted up
}
